;
var muzzley = { };
(function (muzzley, undefined) {

  var readyCallback = function () { };
  document.addEventListener('WebViewJavascriptBridgeReady',
    function (ev) {
      var uniqueId = 1;
      var bridge = ev.bridge;
      bridge.init(function(message, callback) {
        if (callback) {
          return trigger(message, function (success, message, data) {
            callback({ s: success, m: message, d: data });
          });
        }
        return trigger(message);
      });

      muzzley.send = function (data, callback) {
        if (callback) {
          bridge.send(JSON.stringify(data), function (data) {
            return callback(data.s ? null: data.m, data.d);
          });
        } else {
          bridge.send(JSON.stringify(data));
        }
      };

      /**
       * Publish a message to a given namespace
       * 
       * @param  {string}   namespace
       * @param  {object}   payload
       * @param  {Function} callback
       * @return {undefined}
       */
      muzzley.publish = function (namespace, payload, callback) {
        var data;
        if (typeof namespace === 'object') {
          data = {
            ns: namespace.namespace,
            p: namespace.payload,
          };
        } else {
          data = {
            ns: namespace,
            p: payload
          };
        }
        if (callback) {
          bridge.publish(JSON.stringify(data), function (data) {
            return callback(data.s ? null: data.m, data.d);
          });
        } else {
          bridge.publish(JSON.stringify(data));
        }
      };

      /**
       * Subscribe to a channel of a given namespace
       * @param  {object} namespace String with namespace or an object with properties:
       *                       - namespace
       *                       - payload
       * @param  {Function} callback
       * @return {undefined}
       */
      muzzley.subscribe = function (namespace, payload, callback) {
        var data;
        if (typeof namespace === 'object') {
          data = {
            ns: namespace.namespace,
            p: namespace.payload,
          };
        } else {
          data = {
            ns: namespace,
            p: payload
          };
        }
        if (callback) {
          bridge.subscribe(JSON.stringify(data), function (data) {
            return callback(data.s ? null: data.m, data.d);
          });
        } else {
          bridge.subscribe(JSON.stringify(data));
        }
      };

      muzzley.unsubscribe = function (channelId, callback) {
        var data = {
          channelId: channelId
        };
        if (callback) {
          bridge.unsubscribe(JSON.stringify(data), function (data) {
            if (typeof callback !== 'function') return;
            return callback(data.s ? null: data.m, data.d);
          });
        }
      };

      var options = ev.options;
      if (!options) return readyCallback();
      options = JSON.parse(options);
      // Handling the Interfaces ready callback where
      // the channels should be indexed in an object but are not.
      if (options && options.remoteId && options.profileId && options.photoUrl && options.content) {
        options = {
          channels: [options]
        };
      }
      readyCallback(options);
    },
    false
  );

  // Activity Event Listeners
  var listeners = {};
  function addSignalListener(ev, callback) {
    if (!listeners[ev]) {
      listeners[ev] = [];
    }
    listeners[ev].push(callback);
  }
  function removeSignalListener(ev, handler) {
    if (!ev || !listeners[ev]) {
      return;
    }
    if (typeof handler === 'function') {
      for (var i = listeners[ev].length - 1; i >= 0; i--) {
        if (listeners[ev][i] === handler) {
           listeners[ev].splice(i, 1);
        }
      }
    }

    if (!listeners[ev].length || typeof handler !== 'function') {
      delete listeners[ev];
    }
  }

  var channelStorage = {};

  // 
  // Channel Model
  // 
  var Channel = function () {
    this.id = null;
    this._eventHandlers = {};
  };
  Channel.prototype.getId = function () {
    return this.id;
  };
  Channel.prototype.setId = function (id) {
    this.id = id;
  };
  Channel.prototype.on = function (event, handler) {
    this._eventHandlers[event] = this._eventHandlers[event] || [];
    this._eventHandlers[event].push(handler);
  };
  Channel.prototype.off = function (event, handler) {
    var evHandlers = this._eventHandlers[event];
    if (evHandlers && evHandlers.length && evHandlers.indexOf) {
      var index = evHandlers.indexOf(handler);
      if (index > -1) {
        evHandlers.splice(index, 1);
      }
    }
  };
  Channel.prototype.trigger = function (event, data, callback) {
    if (!this._eventHandlers[event]) {
      return;
    }
    $.each(this._eventHandlers[event], function (i, elem){
      elem(data, callback);
    });
  };
  Channel.prototype.unsubscribe = function (callback) {
    var self = this;
    callback = callback || function () {};
    if (this.id === null) {
      return callback(new Error('Channel not yet initialized'));
    }
    muzzley.unsubscribe(this.id, function (err) {
      if (err) {
        self.trigger('error', err);
        return callback(err);
      }

      if (self.id && channelStorage[self.id]) {
        delete channelStorage['c_'+self.id];
      }

      self.trigger('unsubscribe');
      return callback(null);
    });
  };

  // 
  // PubSubMessage Model
  // 
  var PubSubMessage = function (options) {
    this.raw = options.raw;
  };
  PubSubMessage.prototype.getRawMessage = function () {
    return this.raw;
  };
  PubSubMessage.prototype.getRawMessageData = function () {
    return this.getRawMessage().d;
  };
  PubSubMessage.prototype.getNamespace = function () {
    return this.getRawMessageData().ns;
  };
  PubSubMessage.prototype.getPayload = function () {
    return this.getRawMessageData().p;
  };

  /**
   * Trigger activity (signal) or channel-specific (publish) events that are
   * received by the native clients and passed on to the webview js code.
   * @param  {object}   message  The message received by the native client. Either
   *                             a signaling message's `d` field for legacy clients
   *                             or the full muzzley message for newer clients.
   * @param  {object}   data
   * @param  {Function} callback
   * @return {undeifned}
   */
  var trigger = function (message, callback) {

    var ev = null;
    var data = null;

    if (message.h) {
      // New full-message triggering version.
      var channelId = message.h.ch;
      var channel = channelStorage['c_'+channelId];

      if (channel && message.a === 'publish') {
        var pubSubMessage = new PubSubMessage({ raw: message });
        return channel.trigger('message', pubSubMessage, callback);
      } else if (message.a === 'signal') {
        ev = message.d.a;
        data = message.d.d;
      }
    } else {
      // Legacy mobile app version.
      // Signaling message `d` object triggering code.
      ev = message.a;
      data = message.d;
    }

    if (!ev || !listeners[ev]) {
      // Invalid or unsupported message, discard.
      return;
    }
    $.each(listeners[ev], function (i, elem){
      elem(data, callback);
    });
  };

  // Public WebView `muzzley` object
  var lib = {};
  lib.on = function (ev, callback) {
    return $.isFunction(callback) ?
      addSignalListener(ev, callback): false;
  };
  lib.off = function (ev, callback) {
    removeSignalListener(ev, callback);
  };
  lib.ready = function (callback) {
    if (typeof callback !== 'function') return;
    readyCallback = callback;
  };
  // send('setTemperature', { temp: 25, unit: 'C' });
  lib.send = function (action, data, callback) {
    if (muzzley.send) {
      muzzley.send({
          a: action,
          d: data
        },
        callback
      );
    }
  };
  lib.subscribe = function (namespace, payload, callback) {
    if (typeof namespace === 'object') {
      callback = payload;
      payload = null;
    }
    callback = callback || function () {};
    var channel = new Channel();
    if (muzzley.subscribe) {
      muzzley.subscribe(namespace, payload, function (err, data) {
        if (err || !data || !data.channel || !data.channel.id) {
          return callback(err || new Error('Invalid subscription response'));
        }
        var id = data.channel.id;
        channelStorage['c_'+id] = channel;
        channel.setId(id);
        channel.trigger('subscribe');
        return callback(null, channel);
      });
    }
    return channel;
  };
  lib.subscribeIot = function (payload, callback) {
    return lib.subscribe('iot', payload, callback);
  };
  lib.publish = function (namespace, payload, callback) {
    if (typeof namespace === 'object') {
      callback = payload;
      payload = null;
    }
    if (muzzley.publish) {
      muzzley.publish(namespace, payload, callback);
    }
  };
  lib.publishIot = function (payload, callback) {
    return lib.publish('iot', payload, callback);
  };
  muzzley.lib = lib;
})(muzzley);
